"""
Utility functions shared by Volume-4 simulations.

Exposed API (kept minimal & stable):
- load_config(path) / load_yaml(path): read YAML -> dict
- sweep_iter(cfg): yield (b, k, n0, L) from config lists
- seed_all(b, k, n0, L): deterministic RNG seed
- save_csv(path, row): append a dict row to CSV (creates header if new)
"""

from itertools import product
from typing import Any, Dict, Iterator, Tuple
import csv
import os
import random

import numpy as np
import yaml


# ---------------------------------------------------------------------
# Configuration loading
# ---------------------------------------------------------------------
def load_config(path: str) -> Dict[str, Any]:
    """Load a YAML configuration file into a Python dict."""
    if not os.path.exists(path):
        raise FileNotFoundError(f"Configuration file not found: {path}")
    with open(path, "r", encoding="utf-8") as f:
        return yaml.safe_load(f)


def load_yaml(path: str) -> Dict[str, Any]:
    """Alias for load_config (kept for compatibility)."""
    return load_config(path)


# ---------------------------------------------------------------------
# Sweep iterator
# ---------------------------------------------------------------------
def sweep_iter(cfg: Dict[str, Any]) -> Iterator[Tuple[float, float, float, int]]:
    """
    Yield all combinations (b, k, n0, L) from the configuration.

    Requires top-level lists: b_values, k_values, n0_values, L_values.
    """
    try:
        b_vals = cfg["b_values"]
        k_vals = cfg["k_values"]
        n0_vals = cfg["n0_values"]
        L_vals = cfg["L_values"]
    except KeyError as e:
        raise KeyError(f"Missing required config key: {e.args[0]}")
    for b, k, n0, L in product(b_vals, k_vals, n0_vals, L_vals):
        yield float(b), float(k), float(n0), int(L)


# ---------------------------------------------------------------------
# Deterministic RNG seeding
# ---------------------------------------------------------------------
def seed_all(b: float, k: float, n0: float, L: int) -> None:
    """Seed Python and NumPy RNGs deterministically based on (b, k, n0, L)."""
    seed = (
        int(round(float(b) * 1e6))
        ^ int(round(float(k) * 1e6))
        ^ int(round(float(n0) * 1e6))
        ^ int(L)
    ) & 0xFFFFFFFF
    random.seed(seed)
    np.random.seed(seed)


# ---------------------------------------------------------------------
# Lightweight CSV writer
# ---------------------------------------------------------------------
def save_csv(path: str, row: Dict[str, Any]) -> None:
    """
    Append a row (dict) to CSV at `path`. Creates file & header if new.

    - Reuses existing header order if file exists.
    - If creating a new file, header = sorted(row.keys()) for stability.
    - Missing keys are written as blank; extra keys are dropped.
    """
    parent = os.path.dirname(path) or "."
    os.makedirs(parent, exist_ok=True)

    fieldnames = None
    if os.path.exists(path):
        with open(path, "r", encoding="utf-8", newline="") as f_in:
            reader = csv.DictReader(f_in)
            if reader.fieldnames:
                fieldnames = list(reader.fieldnames)

    if fieldnames is None:
        fieldnames = sorted(row.keys())

    write_header = not os.path.exists(path)
    with open(path, "a", encoding="utf-8", newline="") as f_out:
        writer = csv.DictWriter(f_out, fieldnames=fieldnames)
        if write_header:
            writer.writeheader()
        safe_row = {k: row.get(k, None) for k in fieldnames}
        writer.writerow(safe_row)